<?php
/* --------------------------------------------------------------
  ConfigReadService.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

use \StyleEdit\Repositories\ConfigRepository;

/**
 * Class ConfigReadService
 * @package StyleEdit
 */
class ConfigReadService
{
	/**
	 * @var \StyleEdit\Repositories\ConfigRepository
	 */
	private $configRepository;
	
	
	/**
	 * ConfigReadService constructor.
	 *
	 * @param \StyleEdit\Repositories\ConfigRepository $configRepository
	 */
	public function __construct(ConfigRepository $configRepository)
	{
		$this->configRepository = $configRepository;
	}
	
	
	/**
	 * Returns a style configuration identified by its name.
	 *
	 * @param $p_styleName
	 *
	 * @throws \InvalidArgumentException if $p_styleName is not a string
	 * @throws \InvalidArgumentException if $p_styleName is an empty string
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function getStyleConfigByName($p_styleName)
	{
		if(!is_string($p_styleName))
		{
			throw new \InvalidArgumentException('$p_styleName is not a string: ' . gettype($p_styleName));
		}
		
		if($p_styleName === '')
		{
			throw new \InvalidArgumentException('$p_styleName must not be an empty string');
		}
		
		return $this->configRepository->getStyleConfigByName($p_styleName);
	}
	
	
	/**
	 * Returns a boilerplate configuration identified by its name.
	 *
	 * @param $p_boilerplateName
	 *
	 * @throws \InvalidArgumentException if $p_boilerplateName is not a string
	 * @throws \InvalidArgumentException if $p_boilerplateName is an empty string
	 *
	 * @return \StyleEdit\Entities\StoredBoilerplateConfig
	 */
	public function getBoilerplateByName($p_boilerplateName)
	{
		if(!is_string($p_boilerplateName))
		{
			throw new \InvalidArgumentException('$p_boilerplateName is not a string: ' . gettype($p_boilerplateName));
		}
		
		if($p_boilerplateName === '')
		{
			throw new \InvalidArgumentException('$p_boilerplateName must not be an empty string');
		}
		
		return $this->configRepository->getBoilerplateByName($p_boilerplateName);
	}
	
	
	/**
	 * Returns a collection of all style configurations.
	 *
	 * @return \StyleEdit\Collections\StoredConfigCollection
	 */
	public function getStyleConfigs()
	{
		return $this->configRepository->getStyleConfigs();
	}
	
	
	/**
	 * Returns a collection of all boilerplate configurations.
	 *
	 * @return \StyleEdit\Collections\StoredBoilerplateConfigCollection
	 */
	public function getBoilerplates()
	{
		return $this->configRepository->getBoilerplates();
	}
	
	
	/**
	 * Searches for an active style configuration and returns it if found. Otherwise null will be returned.
	 *
	 * @return null|\StyleEdit\Entities\StoredConfig
	 */
	public function findActiveStyleConfig()
	{
		return $this->configRepository->findActiveStyleConfig();
	}
}